<?php
session_start();
if (!isset($_SESSION['admin_logged_in']) || !$_SESSION['admin_logged_in']) {
    header('Location: admin_login.php');
    exit;
}
$admin = $_SESSION['admin_username'];
require '../backend/db.php';

// Handle approve/reject actions
if (isset($_POST['action']) && isset($_POST['recharge_id'])) {
    $recharge_id = intval($_POST['recharge_id']);
    $action = $_POST['action'];
    
    if ($action === 'approve') {
        // Get recharge details
        $recharge_query = $conn->prepare("SELECT r.user_id, r.amount, u.balance FROM recharges r JOIN users u ON r.user_id = u.id WHERE r.id = ?");
        $recharge_query->bind_param("i", $recharge_id);
        $recharge_query->execute();
        $result = $recharge_query->get_result();
        
        if ($row = $result->fetch_assoc()) {
            // Update recharge status
            $update_recharge = $conn->prepare("UPDATE recharges SET status = 'completed' WHERE id = ?");
            $update_recharge->bind_param("i", $recharge_id);
            
            // Update user balance
            $new_balance = $row['balance'] + $row['amount'];
            $update_balance = $conn->prepare("UPDATE users SET balance = ? WHERE id = ?");
            $update_balance->bind_param("di", $new_balance, $row['user_id']);
            
            if ($update_recharge->execute() && $update_balance->execute()) {
                $message = "Recharge approved successfully!";
                $message_type = 'success';
            } else {
                $message = "Failed to approve recharge.";
                $message_type = 'error';
            }
        }
    } elseif ($action === 'reject') {
        // Update recharge status to failed
        $update_recharge = $conn->prepare("UPDATE recharges SET status = 'failed' WHERE id = ?");
        $update_recharge->bind_param("i", $recharge_id);
        
        if ($update_recharge->execute()) {
            $message = "Recharge rejected successfully!";
            $message_type = 'success';
        } else {
            $message = "Failed to reject recharge.";
            $message_type = 'error';
        }
    }
}

// Handle withdrawal approval/rejection
if (isset($_POST['action']) && isset($_POST['withdrawal_id'])) {
    $withdrawal_id = intval($_POST['withdrawal_id']);
    $action = $_POST['action'];
    
    if ($action === 'approve') {
        // Update withdrawal status to approved
        $update_withdrawal = $conn->prepare("UPDATE withdrawals SET status = 'approved' WHERE id = ?");
        $update_withdrawal->bind_param("i", $withdrawal_id);
        
        if ($update_withdrawal->execute()) {
            $message = "Withdrawal approved successfully!";
            $message_type = 'success';
        } else {
            $message = "Failed to approve withdrawal.";
            $message_type = 'error';
        }
    } elseif ($action === 'reject') {
        // Get withdrawal details to refund the amount
        $withdrawal_query = $conn->prepare("SELECT w.user_id, w.amount FROM withdrawals w WHERE w.id = ?");
        $withdrawal_query->bind_param("i", $withdrawal_id);
        $withdrawal_query->execute();
        $result = $withdrawal_query->get_result();
        
        if ($row = $result->fetch_assoc()) {
            // Update withdrawal status to rejected
            $update_withdrawal = $conn->prepare("UPDATE withdrawals SET status = 'rejected' WHERE id = ?");
            $update_withdrawal->bind_param("i", $withdrawal_id);
            
            // Refund the amount to user balance
            $refund_balance = $conn->prepare("UPDATE users SET balance = balance + ? WHERE id = ?");
            $refund_balance->bind_param("di", $row['amount'], $row['user_id']);
            
            if ($update_withdrawal->execute() && $refund_balance->execute()) {
                $message = "Withdrawal rejected and amount refunded successfully!";
                $message_type = 'success';
            } else {
                $message = "Failed to reject withdrawal.";
                $message_type = 'error';
            }
        } else {
            $message = "Withdrawal not found.";
            $message_type = 'error';
        }
    }
}

// Handle admin add recharge
if (isset($_POST['add_recharge']) && isset($_POST['user_id']) && isset($_POST['amount'])) {
    $user_id = intval($_POST['user_id']);
    $amount = floatval($_POST['amount']);
    
    if ($amount > 0) {
        // Generate order number
        $date = date("Ymd");
        $time = time();
        $serial = $date . $time . rand(100000, 999900);
        
        // Insert recharge record
        $stmt = $conn->prepare("INSERT INTO recharges (user_id, amount, order_number, payment_method, status) VALUES (?, ?, ?, 'Admin Recharge', 'completed')");
        $stmt->bind_param("ids", $user_id, $amount, $serial);
        
        if ($stmt->execute()) {
            // Update user balance
            $update_balance = $conn->prepare("UPDATE users SET balance = balance + ? WHERE id = ?");
            $update_balance->bind_param("di", $amount, $user_id);
            
            if ($update_balance->execute()) {
                $message = "Recharge added successfully! Amount: ₹" . number_format($amount, 2);
                $message_type = 'success';
            } else {
                $message = "Failed to update user balance.";
                $message_type = 'error';
            }
        } else {
            $message = "Failed to add recharge.";
            $message_type = 'error';
        }
    } else {
        $message = "Invalid amount.";
        $message_type = 'error';
    }
}

// Handle status update for today's recharge
if (isset($_POST['update_recharge_status']) && isset($_POST['recharge_id']) && isset($_POST['new_status'])) {
    $recharge_id = intval($_POST['recharge_id']);
    $new_status = $_POST['new_status'];
    if (in_array($new_status, ['pending', 'completed', 'failed'])) {
        $conn = new mysqli('localhost', 'okaywin_chikenroad', 'okaywin_chikenroad', 'okaywin_chikenroad');
        $conn->query("UPDATE recharges SET status = '" . $conn->real_escape_string($new_status) . "' WHERE id = $recharge_id");
        $conn->close();
        header('Location: admin_dashboard.php');
        exit;
    }
}

// Date filter logic (single date)
$filter_date = isset($_GET['filter_date']) ? $_GET['filter_date'] : '';
$date_filter_recharge = '';
$date_filter_withdrawal = '';
$date_filter_users = '';
if ($filter_date) {
    $date_filter_recharge = " AND DATE(r.created_at) = '" . $conn->real_escape_string($filter_date) . "' ";
    $date_filter_withdrawal = " AND DATE(w.created_at) = '" . $conn->real_escape_string($filter_date) . "' ";
    $date_filter_users = " AND DATE(created_at) = '" . $conn->real_escape_string($filter_date) . "' ";
}

// Fetch crash_level, random_mode, win_percent, active_mode from game_settings
$crash_level = 5;
$random_mode = 0;
$win_percent = 70;
$active_mode = 'crash';
$gs = $conn->query('SELECT crash_level, random_mode, win_percent, active_mode FROM game_settings WHERE id=1');
if ($gs && $row = $gs->fetch_assoc()) {
    $crash_level = (int)$row['crash_level'];
    $random_mode = (int)$row['random_mode'];
    $win_percent = (int)$row['win_percent'];
    $active_mode = $row['active_mode'];
}

// Fetch recharges with username
$recharges = [];
$rc = $conn->query('SELECT r.id, u.username, r.amount, r.created_at, r.status FROM recharges r JOIN users u ON r.user_id = u.id ORDER BY r.created_at DESC');
if ($rc) {
    while ($row = $rc->fetch_assoc()) {
        $recharges[] = $row;
    }
}

// Fetch pending recharges
$pending_recharges = [];
$pending = $conn->query('SELECT r.id, r.order_number, u.username, r.amount, r.payment_method, r.created_at FROM recharges r JOIN users u ON r.user_id = u.id WHERE r.status = "pending" ORDER BY r.created_at DESC');
if ($pending) {
    while ($row = $pending->fetch_assoc()) {
        $pending_recharges[] = $row;
    }
}

// Fetch withdrawal requests
$withdrawals = [];
$withdrawal_query = $conn->query('SELECT w.id, u.username, w.amount, w.bank_name, w.account_number, w.ifsc_code, w.account_holder, w.status, w.created_at FROM withdrawals w JOIN users u ON w.user_id = u.id ORDER BY w.created_at DESC');
if ($withdrawal_query) {
    while ($row = $withdrawal_query->fetch_assoc()) {
        $withdrawals[] = $row;
    }
}

// Fetch pending withdrawals
$pending_withdrawals = [];
$pending_withdrawal_query = $conn->query('SELECT w.id, u.username, w.amount, w.bank_name, w.account_number, w.ifsc_code, w.account_holder, w.created_at FROM withdrawals w JOIN users u ON w.user_id = u.id WHERE w.status = "pending" ORDER BY w.created_at DESC');
if ($pending_withdrawal_query) {
    while ($row = $pending_withdrawal_query->fetch_assoc()) {
        $pending_withdrawals[] = $row;
    }
}

// Fetch referral statistics
$referral_stats = [];
$ref_stats_query = $conn->query('
    SELECT 
        u.username,
        u.referral_code,
        u.total_referrals,
        u.total_referral_earnings,
        COUNT(r.id) as total_referrals_count,
        SUM(rr.amount) as total_earnings_sum
    FROM users u
    LEFT JOIN referrals r ON u.id = r.referrer_id
    LEFT JOIN referral_rewards rr ON u.id = rr.referrer_id
    WHERE u.total_referrals > 0
    GROUP BY u.id
    ORDER BY u.total_referral_earnings DESC
    LIMIT 20
');
if ($ref_stats_query) {
    while ($row = $ref_stats_query->fetch_assoc()) {
        $referral_stats[] = $row;
    }
}

// Get total referral statistics
$total_ref_stats = $conn->query('
    SELECT 
        COUNT(DISTINCT referrer_id) as total_referrers,
        COUNT(DISTINCT referred_id) as total_referred,
        SUM(amount) as total_referral_payouts
    FROM referral_rewards 
    WHERE status = "paid"
')->fetch_assoc();

// --- SUMMARY BOXES CALCULATION ---
// Total Recharge (completed)
$total_recharge = $conn->query("SELECT SUM(amount) as total FROM recharges WHERE status='completed'" . ($filter_date ? " AND DATE(created_at) = '" . $conn->real_escape_string($filter_date) . "'" : ""))->fetch_assoc()["total"] ?? 0;
// Total Withdrawal (approved)
$total_withdrawal = $conn->query("SELECT SUM(amount) as total FROM withdrawals WHERE status='approved'" . ($filter_date ? " AND DATE(created_at) = '" . $conn->real_escape_string($filter_date) . "'" : ""))->fetch_assoc()["total"] ?? 0;
// Total Users
$total_users = $conn->query("SELECT COUNT(*) as total FROM users WHERE 1=1" . $date_filter_users)->fetch_assoc()["total"] ?? 0;
// Total Profit (recharge - withdrawal)
$total_profit = $total_recharge - $total_withdrawal;

// --- TODAY'S SUMMARY CALCULATION ---
$today = date('Y-m-d');
$today_recharge = $conn->query("SELECT SUM(amount) as total FROM recharges WHERE status='completed' AND DATE(created_at) = '$today'")->fetch_assoc()["total"] ?? 0;
$today_withdrawal = $conn->query("SELECT SUM(amount) as total FROM withdrawals WHERE status='approved' AND DATE(created_at) = '$today'")->fetch_assoc()["total"] ?? 0;

// Filtered recharges
$recharges = [];
$rc = $conn->query("SELECT r.id, u.username, r.amount, r.created_at, r.status FROM recharges r JOIN users u ON r.user_id = u.id WHERE 1=1 $date_filter_recharge ORDER BY r.created_at DESC");
if ($rc) {
    while ($row = $rc->fetch_assoc()) {
        $recharges[] = $row;
    }
}
// Filtered withdrawals
$withdrawals = [];
$withdrawal_query = $conn->query("SELECT w.id, u.username, w.amount, w.bank_name, w.account_number, w.ifsc_code, w.account_holder, w.status, w.created_at FROM withdrawals w JOIN users u ON w.user_id = u.id WHERE 1=1 $date_filter_withdrawal ORDER BY w.created_at DESC");
if ($withdrawal_query) {
    while ($row = $withdrawal_query->fetch_assoc()) {
        $withdrawals[] = $row;
    }
}
// Filtered users
$users = [];
$result = $conn->query('SELECT id, username, email, balance, created_at FROM users WHERE 1=1' . $date_filter_users . ' ORDER BY id DESC');
if ($result) {
    while ($row = $result->fetch_assoc()) {
        $users[] = $row;
    }
}

// --- TODAY'S RECHARGES TABLE ---
$today_recharges = [];
$trc = $conn->query("SELECT u.username, r.amount, r.status, r.created_at FROM recharges r JOIN users u ON r.user_id = u.id WHERE DATE(r.created_at) = '$today' ORDER BY r.created_at DESC");
if ($trc) {
    while ($row = $trc->fetch_assoc()) {
        $today_recharges[] = $row;
    }
}

$conn->close();

// Show status message if redirected
$gs_msg = '';
if (isset($_GET['gs']) && $_GET['gs'] === 'ok') $gs_msg = '<div class="register-success">Game settings updated!</div>';
if (isset($_GET['gs']) && $_GET['gs'] === 'err') $gs_msg = '<div class="register-error">Failed to update settings.</div>';

// Show action message
$action_msg = '';
if (isset($message)) {
    $action_msg = '<div class="register-' . $message_type . '">' . $message . '</div>';
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Admin Dashboard - Chicken Road</title>
    <link rel="stylesheet" href="style.css">
    <link href="https://fonts.googleapis.com/css2?family=Inter:wght@400;600;700&display=swap" rel="stylesheet">
    <style>
        body {
            background: linear-gradient(135deg, #181c2b 0%, #23243a 100%);
            font-family: 'Inter', Arial, sans-serif;
            margin: 0;
            display: flex;
        }
        .admin-sidebar {
            width: 220px;
            min-width: 180px;
            background: rgba(26,29,37,0.85);
            height: 100vh;
            position: fixed;
            left: 0;
            top: 0;
            display: flex;
            flex-direction: column;
            align-items: stretch;
            box-shadow: 2px 0 24px #00ffe733, 0 0 0 1.5px #00ffe7;
            z-index: 10;
            backdrop-filter: blur(16px) saturate(1.2);
            border-right: 2.5px solid #00ffe7;
            animation: sidebar-slide-in 0.7s cubic-bezier(.6,-0.28,.74,.05);
        }
        @keyframes sidebar-slide-in {
            0% { transform: translateX(-100%); opacity: 0; }
            100% { transform: translateX(0); opacity: 1; }
        }
        .admin-sidebar-header {
            padding: 32px 0 18px 0;
            display: flex;
            flex-direction: column;
            align-items: center;
            border-bottom: 1.5px solid #00ffe7;
        }
        .admin-avatar {
            width: 62px;
            height: 62px;
            border-radius: 50%;
            background: #383f4e url('https://api.dicebear.com/7.x/bottts/svg?seed=Admin') center/cover no-repeat;
            margin-bottom: 10px;
            box-shadow: 0 0 16px #00ffe7cc;
            border: 2.5px solid #00ffe7;
        }
        .admin-username {
            color: #ffe066;
            font-size: 1.18rem;
            font-weight: 700;
            letter-spacing: 0.5px;
            text-shadow: 0 2px 12px #00ffe7aa;
        }
        .admin-sidebar-menu {
            flex: 1;
            display: flex;
            flex-direction: column;
            gap: 8px;
            margin-top: 24px;
        }
        .admin-sidebar-link {
            display: flex;
            align-items: center;
            gap: 14px;
            color: #fff;
            font-size: 1.13rem;
            padding: 13px 30px;
            text-decoration: none;
            transition: background 0.18s, color 0.18s, box-shadow 0.18s;
            border-left: 4px solid transparent;
            border-radius: 0 18px 18px 0;
            position: relative;
            overflow: hidden;
        }
        .admin-sidebar-link.active, .admin-sidebar-link:hover {
            background: linear-gradient(90deg, #00ffe7 0%, #23243a 100%);
            color: #181c2b;
            border-left: 4px solid #00ffe7;
            box-shadow: 0 2px 16px #00ffe733;
        }
        .admin-sidebar-link span {
            transition: transform 0.2s;
        }
        .admin-sidebar-link:hover span {
            transform: scale(1.13) rotate(-4deg);
            filter: drop-shadow(0 0 8px #00ffe7);
        }
        .admin-sidebar-footer {
            padding: 18px 0 12px 0;
            text-align: center;
            color: #fff;
            font-size: 1.05rem;
            border-top: 1.5px solid #00ffe7;
            letter-spacing: 0.5px;
        }
        .admin-main {
            margin-left: 220px;
            width: 100%;
            padding: 40px 0 0 0;
            display: flex;
            flex-direction: column;
            align-items: center;
            animation: fade-in 0.7s cubic-bezier(.6,-0.28,.74,.05);
        }
        @keyframes fade-in {
            0% { opacity: 0; transform: translateY(30px); }
            100% { opacity: 1; transform: translateY(0); }
        }
        .admin-title {
            font-size: 2.2rem;
            color: #00ffe7;
            font-weight: bold;
            margin-bottom: 18px;
            text-align: center;
            letter-spacing: 1.5px;
            text-shadow: 0 2px 24px #00ffe7aa;
            animation: pulse-glow 1.5s infinite alternate;
        }
        @keyframes pulse-glow {
            0% { text-shadow: 0 2px 24px #00ffe7aa; }
            100% { text-shadow: 0 4px 36px #00ffe7cc; }
        }
        .admin-welcome {
            color: #fff;
            font-size: 1.2rem;
            margin-bottom: 24px;
            text-align: center;
        }
        .admin-btn {
            width: 100%;
            background: #e11d48;
            color: #fff;
            border: none;
            border-radius: 8px;
            padding: 12px 0;
            font-size: 1.1rem;
            font-weight: 700;
            cursor: pointer;
            transition: background 0.2s;
        }
        .admin-btn:hover { background: #b91c1c; }
        .users-table {
            width: 100%;
            border-collapse: collapse;
            margin-top: 18px;
            background: #232834;
            border-radius: 8px;
            overflow: hidden;
        }
        .users-table th, .users-table td {
            padding: 10px 14px;
            text-align: left;
        }
        .users-table th {
            background: #383f4e;
            color: #ffe066;
            font-weight: 700;
        }
        .users-table tr:nth-child(even) {
            background: #2c3240;
        }
        .users-table tr:nth-child(odd) {
            background: #232834;
        }
        .users-table td {
            color: #fff;
            font-size: 1rem;
        }
        .users-table tr:hover {
            background: #383f4e;
        }
        .users-table tr {
            transition: background 0.25s, box-shadow 0.25s;
        }
        .users-table tr:hover {
            background: #00ffe733 !important;
            box-shadow: 0 2px 12px #00ffe7cc;
        }
        .users-section-title {
            color: #ffe066;
            font-size: 1.3rem;
            font-weight: 600;
            margin-bottom: 8px;
        }
        .settings-section-title {
            color: #ffe066;
            font-size: 1.3rem;
            font-weight: 600;
            margin-bottom: 8px;
        }
        .settings-placeholder {
            color: #fff;
            font-size: 1.05rem;
            margin-bottom: 10px;
        }
        .settings-form label {
            color: #fff;
            font-size: 1.1rem;
            font-weight: 500;
        }
        .settings-form select, .settings-form input[type=number] {
            width: 80px;
            padding: 8px 10px;
            margin: 8px 0 18px 0;
            border-radius: 6px;
            border: none;
            background: #383f4e;
            color: #fff;
            font-size: 1rem;
        }
        .settings-form button {
            background: #22c55e;
            color: #fff;
            border: none;
            border-radius: 8px;
            padding: 10px 28px;
            font-size: 1.1rem;
            font-weight: 700;
            cursor: pointer;
            transition: background 0.2s;
        }
        .settings-form button:hover { background: #16a34a; }
        .register-success {
            color: #22c55e;
            font-weight: 600;
            margin-bottom: 10px;
            text-align: center;
        }
        .register-error {
            color: #e11d48;
            font-weight: 600;
            margin-bottom: 10px;
            text-align: center;
        }
        @media (max-width: 900px) {
            .admin-sidebar { width: 60px; min-width: 60px; }
            .admin-main { margin-left: 60px; }
            .admin-sidebar-link span { display: none; }
            .admin-sidebar-link { padding: 12px 10px; justify-content: center; }
        }
        .admin-section { display: none; }
        .admin-section.active { display: block; }
        .action-buttons {
            display: flex;
            gap: 8px;
        }
        .approve-btn {
            background: #22c55e;
            color: #fff;
            border: none;
            border-radius: 6px;
            padding: 6px 12px;
            font-size: 0.9rem;
            cursor: pointer;
            transition: background 0.2s;
        }
        .approve-btn:hover {
            background: #16a34a;
        }
        .reject-btn {
            background: #e11d48;
            color: #fff;
            border: none;
            border-radius: 6px;
            padding: 6px 12px;
            font-size: 0.9rem;
            cursor: pointer;
            transition: background 0.2s;
        }
        .reject-btn:hover {
            background: #be123c;
        }
        .status-badge {
            display: inline-block;
            padding: 5px 14px;
            border-radius: 16px;
            font-size: 1rem;
            font-weight: 700;
            text-transform: capitalize;
            letter-spacing: 0.5px;
            box-shadow: 0 2px 8px #0001;
            margin: 0 2px;
        }
        .status-pending { background: #fbbf24; color: #92400e; border: 1.5px solid #f59e42; }
        .status-completed { background: #22c55e; color: #fff; border: 1.5px solid #16a34a; }
        .status-failed { background: #ef4444; color: #fff; border: 1.5px solid #b91c1c; }
        .status-select {
            background: #23243a;
            color: #fff;
            border: 1.5px solid #383f4e;
            border-radius: 8px;
            padding: 5px 10px;
            font-size: 1rem;
            font-weight: 600;
        }
        .status-update-btn {
            background: linear-gradient(135deg, #22c55e 0%, #3b82f6 100%);
            color: #fff;
            border: none;
            border-radius: 8px;
            padding: 6px 14px;
            font-size: 1rem;
            font-weight: 700;
            cursor: pointer;
            transition: background 0.2s, box-shadow 0.2s;
            box-shadow: 0 2px 8px #22c55e22;
        }
        .status-update-btn:hover {
            background: linear-gradient(135deg, #3b82f6 0%, #22c55e 100%);
            box-shadow: 0 4px 16px #3b82f622;
        }
        .add-recharge-form {
            background: rgba(51, 65, 85, 0.8);
            padding: 20px;
            border-radius: 12px;
            margin-bottom: 20px;
        }
        .add-recharge-form label {
            color: #f8fafc;
            font-size: 1rem;
            margin-bottom: 8px;
            display: block;
            font-weight: 600;
        }
        .add-recharge-form select, .add-recharge-form input {
            width: 100%;
            padding: 12px;
            border: 2px solid #475569;
            border-radius: 8px;
            background: #1e293b;
            color: #f8fafc;
            font-size: 1rem;
            margin-bottom: 16px;
        }
        .add-recharge-form select:focus, .add-recharge-form input:focus {
            outline: none;
            border-color: #22c55e;
        }
        .add-recharge-btn {
            background: linear-gradient(135deg, #22c55e 0%, #16a34a 100%);
            color: #fff;
            border: none;
            padding: 12px 24px;
            border-radius: 8px;
            font-size: 1rem;
            font-weight: 600;
            cursor: pointer;
            transition: all 0.3s ease;
        }
        .add-recharge-btn:hover {
            transform: translateY(-2px);
            box-shadow: 0 8px 20px rgba(34, 197, 94, 0.3);
        }
        .user-balance {
            color: #94a3b8;
            font-size: 0.9rem;
            margin-top: 4px;
        }
        .admin-summary-row {
            display: flex;
            gap: 28px;
            margin-bottom: 32px;
            justify-content: center;
            flex-wrap: wrap;
        }
        .admin-summary-box {
            background: linear-gradient(135deg, #23243a 60%, #2c3240 100%);
            border-radius: 18px;
            box-shadow: 0 4px 24px #00ffe733, 0 0 0 2px #00ffe7 inset;
            padding: 32px 40px 22px 40px;
            min-width: 190px;
            text-align: center;
            position: relative;
            overflow: hidden;
            transition: transform 0.18s, box-shadow 0.18s;
            border-top: 4px solid #00ffe7;
            animation: fade-in 0.7s cubic-bezier(.6,-0.28,.74,.05);
        }
        .admin-summary-box:hover {
            transform: scale(1.04) translateY(-4px);
            box-shadow: 0 8px 32px #00ffe7cc, 0 0 0 2px #00ffe7 inset;
        }
        .summary-title {
            color: #ffe066;
            font-size: 1.18rem;
            font-weight: 700;
            margin-bottom: 10px;
            letter-spacing: 0.5px;
            text-shadow: 0 2px 12px #00ffe7aa;
        }
        .summary-value {
            font-size: 2.3rem;
            font-weight: bold;
            color: #fff;
            text-shadow: 0 2px 18px #00ffe7cc;
            animation: pulse-glow 1.5s infinite alternate;
        }
        .summary-recharge { border-top: 4px solid #22c55e; }
        .summary-withdrawal { border-top: 4px solid #e11d48; }
        .summary-users { border-top: 4px solid #3b82f6; }
        .summary-profit { border-top: 4px solid #ffe066; }
        .admin-datefilter-row {
            width: 100%;
            display: flex;
            justify-content: center;
            margin-bottom: 18px;
        }
        .admin-datefilter-form {
            display: flex;
            gap: 16px;
            align-items: center;
            background: #23243a;
            padding: 16px 28px;
            border-radius: 12px;
            box-shadow: 0 2px 12px #0002;
        }
        .admin-datefilter-form label {
            color: #ffe066;
            font-weight: 600;
            font-size: 1rem;
        }
        .admin-datefilter-form input[type=date] {
            background: #2c3240;
            color: #fff;
            border: 1.5px solid #383f4e;
            border-radius: 6px;
            padding: 6px 10px;
            font-size: 1rem;
            margin-left: 6px;
        }
        .admin-datefilter-form .admin-btn {
            padding: 8px 18px;
            font-size: 1rem;
            margin-left: 8px;
        }
        .admin-reset-link {
            color: #e11d48;
            font-weight: 600;
            margin-left: 10px;
            text-decoration: none;
            font-size: 1rem;
        }
        .admin-reset-link:hover { text-decoration: underline; }
        @media (max-width: 900px) {
            .admin-summary-row { flex-direction: column; gap: 18px; align-items: center; }
            .admin-datefilter-form { flex-direction: column; gap: 10px; align-items: flex-start; }
        }
        .admin-datetime-row {
            width: 100%;
            display: flex;
            justify-content: center;
            align-items: center;
            gap: 12px;
            margin-bottom: 10px;
        }
        .admin-datetime-label {
            color: #ffe066;
            font-weight: 600;
            font-size: 1.08rem;
        }
        .admin-datetime-value {
            color: #fff;
            font-size: 1.08rem;
            font-family: 'Segoe UI', Arial, sans-serif;
            letter-spacing: 0.5px;
        }
        .summary-today-recharge { border-top: 4px solid #38bdf8; }
        .summary-today-withdrawal { border-top: 4px solid #fbbf24; }
        .admin-today-recharge-table-box {
            width: 100%;
            max-width: 700px;
            margin: 32px auto 0 auto;
            background: #23243a;
            border-radius: 14px;
            box-shadow: 0 2px 12px #0002;
            padding: 24px 18px 18px 18px;
        }
        .admin-btn, .add-recharge-btn, .status-update-btn {
            background: linear-gradient(90deg, #00ffe7 0%, #3b82f6 100%);
            color: #181c2b;
            border: none;
            border-radius: 10px;
            padding: 12px 0;
            font-size: 1.13rem;
            font-weight: 700;
            cursor: pointer;
            transition: background 0.2s, box-shadow 0.2s, transform 0.15s;
            box-shadow: 0 2px 12px #00ffe733;
            letter-spacing: 0.5px;
            outline: none;
            position: relative;
            overflow: hidden;
        }
        .admin-btn:active, .add-recharge-btn:active, .status-update-btn:active {
            transform: scale(0.97);
        }
        .admin-btn:hover, .add-recharge-btn:hover, .status-update-btn:hover {
            background: linear-gradient(90deg, #3b82f6 0%, #00ffe7 100%);
            box-shadow: 0 4px 24px #00ffe7cc;
        }
    </style>
</head>
<body>
    <div class="admin-sidebar">
        <div class="admin-sidebar-header">
            <div class="admin-avatar"></div>
            <div class="admin-username"><?php echo htmlspecialchars($admin); ?></div>
        </div>
        <div class="admin-sidebar-menu">
            <a href="#dashboard" class="admin-sidebar-link active" data-section="dashboard">🏠 <span>Dashboard</span></a>
            <a href="#live-users" class="admin-sidebar-link" data-section="live-users">👥 <span>Live Users</span></a>
            <a href="#add-recharge" class="admin-sidebar-link" data-section="add-recharge">➕ <span>Add Recharge</span></a>
            <a href="#pending-recharges" class="admin-sidebar-link" data-section="pending-recharges">⏳ <span>Pending Recharges</span></a>
            <a href="#withdrawal-requests" class="admin-sidebar-link" data-section="withdrawal-requests">💰 <span>Withdrawal Requests</span></a>
            <a href="#game-settings" class="admin-sidebar-link" data-section="game-settings">⚙️ <span>Game Settings</span></a>
            <a href="#referral-stats" class="admin-sidebar-link" data-section="referral-stats">🎯 <span>Referral Stats</span></a>
            <a href="#add-admin" class="admin-sidebar-link" data-section="add-admin">➕ <span>Add Admin</span></a>
            <a href="admin_logout.php" class="admin-sidebar-link">🚪 <span>Logout</span></a>
        </div>
        <div class="admin-sidebar-footer">
            <span>Chicken Road Admin</span>
        </div>
    </div>
    <div class="admin-main">
        <div class="admin-box admin-section active" id="dashboard">
            <div class="admin-title">Admin Dashboard</div>
            <div class="admin-welcome">Welcome, <?php echo htmlspecialchars($admin); ?>!</div>
            <div class="admin-datetime-row">
                <span class="admin-datetime-label">Current Date & Time:</span>
                <span class="admin-datetime-value"><?php echo date('d M Y, h:i A'); ?></span>
            </div>
            <div class="admin-datefilter-row">
                <form method="get" class="admin-datefilter-form">
                    <label>Date: <input type="date" name="filter_date" value="<?php echo htmlspecialchars($filter_date); ?>" required></label>
                    <button type="submit" class="admin-btn">Show</button>
                    <?php if ($filter_date): ?><a href="admin_dashboard.php" class="admin-reset-link">Reset</a><?php endif; ?>
                </form>
            </div>
            <div class="admin-summary-row">
                <div class="admin-summary-box summary-today-recharge">
                    <div class="summary-title">Today Recharge</div>
                    <div class="summary-value">₹<?php echo number_format($today_recharge, 2); ?></div>
                </div>
                <div class="admin-summary-box summary-today-withdrawal">
                    <div class="summary-title">Today Withdrawal</div>
                    <div class="summary-value">₹<?php echo number_format($today_withdrawal, 2); ?></div>
                </div>
                <div class="admin-summary-box summary-recharge">
                    <div class="summary-title">Total Recharge</div>
                    <div class="summary-value">₹<?php echo number_format($total_recharge, 2); ?></div>
                </div>
                <div class="admin-summary-box summary-withdrawal">
                    <div class="summary-title">Total Withdrawal</div>
                    <div class="summary-value">₹<?php echo number_format($total_withdrawal, 2); ?></div>
                </div>
            </div>
            <div class="admin-today-recharge-table-box">
                <div class="users-section-title">Today's Recharges</div>
                <table class="users-table">
                    <tr>
                        <th>Username</th>
                        <th>Amount (₹)</th>
                        <th>Status</th>
                        <th>Date/Time</th>
                    </tr>
                    <?php if (empty($today_recharges)): ?>
                        <tr><td colspan="4" style="text-align:center;color:#94a3b8;">No recharges today.</td></tr>
                    <?php else: ?>
                        <?php foreach ($today_recharges as $tr): ?>
                            <tr>
                                <td><?php echo htmlspecialchars($tr['username']); ?></td>
                                <td><?php echo number_format($tr['amount'], 2); ?></td>
                                <td><span class="status-badge status-<?php echo $tr['status']; ?>"><?php echo ucfirst($tr['status']); ?></span></td>
                                <td><?php echo date('d M Y, h:i A', strtotime($tr['created_at'])); ?></td>
                            </tr>
                        <?php endforeach; ?>
                    <?php endif; ?>
                </table>
            </div>
        </div>
        
        <div class="admin-box admin-section" id="live-users" style="min-width:500px;">
            <div class="users-section-title">Live Users</div>
            <table class="users-table">
                <tr>
                    <th>ID</th>
                    <th>Username</th>
                    <th>Email</th>
                    <th>Balance</th>
                    <th>Created At</th>
                </tr>
                <?php foreach ($users as $user): ?>
                <tr>
                    <td><?php echo htmlspecialchars($user['id']); ?></td>
                    <td><?php echo htmlspecialchars($user['username']); ?></td>
                    <td><?php echo htmlspecialchars($user['email']); ?></td>
                    <td>₹<?php echo number_format($user['balance'], 2); ?></td>
                    <td><?php echo htmlspecialchars($user['created_at']); ?></td>
                </tr>
                <?php endforeach; ?>
            </table>
        </div>
        
        <div class="admin-box admin-section" id="recharge" style="min-width:500px;">
            <div class="users-section-title">Recharge History</div>
            <?php echo $action_msg; ?>
            <table class="users-table">
                <tr>
                    <th>ID</th>
                    <th>Username</th>
                    <th>Amount (₹)</th>
                    <th>Status</th>
                    <th>Date/Time</th>
                </tr>
                <?php foreach ($recharges as $r): ?>
                <tr>
                    <td><?php echo htmlspecialchars($r['id']); ?></td>
                    <td><?php echo htmlspecialchars($r['username']); ?></td>
                    <td><?php echo htmlspecialchars($r['amount']); ?></td>
                    <td>
                        <span class="status-badge status-<?php echo $r['status'] ?? 'pending'; ?>">
                            <?php echo ucfirst($r['status'] ?? 'pending'); ?>
                        </span>
                    </td>
                    <td><?php echo htmlspecialchars($r['created_at']); ?></td>
                </tr>
                <?php endforeach; ?>
            </table>
        </div>
        
        <div class="admin-box admin-section" id="add-recharge" style="min-width:500px;">
            <div class="users-section-title">Add Recharge</div>
            <?php echo $action_msg; ?>
            <div class="add-recharge-form">
                <form method="POST">
                    <label for="user_id">Select User:</label>
                    <select name="user_id" id="user_id" required>
                        <option value="">Choose a user...</option>
                        <?php foreach ($users as $user): ?>
                            <option value="<?php echo $user['id']; ?>">
                                <?php echo htmlspecialchars($user['username']); ?> (₹<?php echo number_format($user['balance'], 2); ?>)
                            </option>
                        <?php endforeach; ?>
                    </select>
                    
                    <label for="amount">Amount (₹):</label>
                    <input type="number" name="amount" id="amount" min="1" step="0.01" placeholder="Enter amount" required>
                    
                    <button type="submit" name="add_recharge" class="add-recharge-btn">💳 Add Recharge</button>
                </form>
            </div>
        </div>
        
        <div class="admin-box admin-section" id="pending-recharges" style="min-width:500px;">
            <div class="users-section-title">Pending Recharge Requests</div>
            <?php echo $action_msg; ?>
            <?php if (empty($pending_recharges)): ?>
                <p style="color: #94a3b8; text-align: center; padding: 20px;">No pending recharge requests.</p>
            <?php else: ?>
                <table class="users-table">
                    <tr>
                        <th>Order #</th>
                        <th>Username</th>
                        <th>Amount (₹)</th>
                        <th>Payment Method</th>
                        <th>Date/Time</th>
                        <th>Actions</th>
                    </tr>
                    <?php foreach ($pending_recharges as $pr): ?>
                    <tr>
                        <td><?php echo htmlspecialchars($pr['order_number'] ?? $pr['id']); ?></td>
                        <td><?php echo htmlspecialchars($pr['username']); ?></td>
                        <td><?php echo htmlspecialchars($pr['amount']); ?></td>
                        <td><?php echo htmlspecialchars($pr['payment_method'] ?? 'N/A'); ?></td>
                        <td><?php echo htmlspecialchars($pr['created_at']); ?></td>
                        <td>
                            <form method="POST" style="display: inline;">
                                <input type="hidden" name="recharge_id" value="<?php echo $pr['id']; ?>">
                                <button type="submit" name="action" value="approve" class="approve-btn">✅ Approve</button>
                                <button type="submit" name="action" value="reject" class="reject-btn">❌ Reject</button>
                            </form>
                        </td>
                    </tr>
                    <?php endforeach; ?>
                </table>
            <?php endif; ?>
        </div>
        
        <div class="admin-box admin-section" id="game-settings" style="min-width:500px;">
            <div class="settings-section-title">Game Settings</div>
            <?php echo $gs_msg; ?>
            <form class="settings-form" method="post" action="game_settings.php" id="gameSettingsForm">
                <div style="margin-bottom:18px;font-weight:600;color:#ffe066;">Crash Logic Mode (Only one can be active):</div>
                <label style="display:flex;align-items:center;margin-bottom:8px;">
                    <input type="radio" name="active_mode" value="crash" id="mode_crash" <?php if($active_mode==='crash') echo 'checked'; ?>>
                    <span style="margin-left:8px;">Crash Level</span>
                </label>
                <div style="margin-left:24px;margin-bottom:10px;">
                    <label for="crash_level">Crash Level (1-8):</label>
                    <select id="crash_level" name="crash_level" <?php if($active_mode!=='crash') echo 'disabled'; ?>>
                        <?php for ($i=1; $i<=8; $i++): ?>
                            <option value="<?php echo $i; ?>" <?php if ($crash_level==$i) echo 'selected'; ?>><?php echo $i; ?></option>
                        <?php endfor; ?>
                    </select>
                </div>
                <label style="display:flex;align-items:center;margin-bottom:8px;">
                    <input type="radio" name="active_mode" value="random" id="mode_random" <?php if($active_mode==='random') echo 'checked'; ?>>
                    <span style="margin-left:8px;">Random Mode</span>
                </label>
                <div style="margin-left:24px;margin-bottom:10px;">
                    <span style="color:#94a3b8;">(Chicken will crash at a random level)</span>
                </div>
                <label style="display:flex;align-items:center;margin-bottom:8px;">
                    <input type="radio" name="active_mode" value="win_percent" id="mode_win_percent" <?php if($active_mode==='win_percent') echo 'checked'; ?>>
                    <span style="margin-left:8px;">Winning Percentage</span>
                </label>
                <div style="margin-left:24px;margin-bottom:10px;">
                    <label for="win_percent">Winning Percentage (1-100):</label>
                    <input type="number" id="win_percent" name="win_percent" min="1" max="100" value="<?php echo $win_percent; ?>" style="margin-left:8px;width:70px;" <?php if($active_mode!=='win_percent') echo 'disabled'; ?>> %
                </div>
                <button type="submit">Save</button>
            </form>
            <script>
                // Enable/disable inputs based on selected mode
                const crashRadio = document.getElementById('mode_crash');
                const randomRadio = document.getElementById('mode_random');
                const winRadio = document.getElementById('mode_win_percent');
                const crashLevel = document.getElementById('crash_level');
                const winPercent = document.getElementById('win_percent');
                function updateModeInputs() {
                    crashLevel.disabled = !crashRadio.checked;
                    winPercent.disabled = !winRadio.checked;
                }
                crashRadio.addEventListener('change', updateModeInputs);
                randomRadio.addEventListener('change', updateModeInputs);
                winRadio.addEventListener('change', updateModeInputs);
                window.addEventListener('DOMContentLoaded', updateModeInputs);
            </script>
        </div>
        
        <div class="admin-box admin-section" id="withdrawal-requests" style="min-width:500px;">
            <div class="users-section-title">Withdrawal Requests</div>
            <?php echo $action_msg; ?>
            <?php if (empty($withdrawals)): ?>
                <p style="color: #94a3b8; text-align: center; padding: 20px;">No withdrawal requests found.</p>
            <?php else: ?>
                <table class="users-table">
                    <tr>
                        <th>ID</th>
                        <th>Username</th>
                        <th>Amount (₹)</th>
                        <th>Bank Details</th>
                        <th>Status</th>
                        <th>Date/Time</th>
                        <th>Actions</th>
                    </tr>
                    <?php foreach ($withdrawals as $w): ?>
                    <tr>
                        <td><?php echo htmlspecialchars($w['id']); ?></td>
                        <td><?php echo htmlspecialchars($w['username']); ?></td>
                        <td>₹<?php echo number_format($w['amount'], 2); ?></td>
                        <td>
                            <div style="font-size: 0.9rem;">
                                <div><strong>Bank:</strong> <?php echo htmlspecialchars($w['bank_name']); ?></div>
                                <div><strong>Account:</strong> <?php echo htmlspecialchars($w['account_holder']); ?></div>
                                <div><strong>IFSC:</strong> <?php echo htmlspecialchars($w['ifsc_code']); ?></div>
                            </div>
                        </td>
                        <td>
                            <span class="status-badge status-<?php echo $w['status']; ?>">
                                <?php echo ucfirst($w['status']); ?>
                            </span>
                        </td>
                        <td><?php echo htmlspecialchars($w['created_at']); ?></td>
                        <td>
                            <?php if ($w['status'] === 'pending'): ?>
                                <form method="POST" style="display: inline;">
                                    <input type="hidden" name="withdrawal_id" value="<?php echo $w['id']; ?>">
                                    <button type="submit" name="action" value="approve" class="approve-btn">✅ Approve</button>
                                    <button type="submit" name="action" value="reject" class="reject-btn">❌ Reject</button>
                                </form>
                            <?php else: ?>
                                <span style="color: #94a3b8; font-size: 0.9rem;">
                                    <?php echo $w['status'] === 'approved' ? 'Approved' : 'Rejected'; ?>
                                </span>
                            <?php endif; ?>
                        </td>
                    </tr>
                    <?php endforeach; ?>
                </table>
            <?php endif; ?>
        </div>
        
        <div class="admin-box admin-section" id="referral-stats" style="min-width:600px;">
            <div class="users-section-title">Referral Statistics</div>
            
            <!-- Total Stats -->
            <div style="display: grid; grid-template-columns: repeat(auto-fit, minmax(200px, 1fr)); gap: 16px; margin-bottom: 24px;">
                <div style="background: rgba(34, 197, 94, 0.1); padding: 16px; border-radius: 8px; text-align: center;">
                    <div style="font-size: 1.5rem; font-weight: bold; color: #22c55e;"><?php echo $total_ref_stats['total_referrers'] ?? 0; ?></div>
                    <div style="font-size: 0.9rem; color: #94a3b8;">Total Referrers</div>
                </div>
                <div style="background: rgba(59, 130, 246, 0.1); padding: 16px; border-radius: 8px; text-align: center;">
                    <div style="font-size: 1.5rem; font-weight: bold; color: #3b82f6;"><?php echo $total_ref_stats['total_referred'] ?? 0; ?></div>
                    <div style="font-size: 0.9rem; color: #94a3b8;">Total Referred</div>
                </div>
                <div style="background: rgba(251, 191, 36, 0.1); padding: 16px; border-radius: 8px; text-align: center;">
                    <div style="font-size: 1.5rem; font-weight: bold; color: #fbbf24;">₹<?php echo number_format($total_ref_stats['total_referral_payouts'] ?? 0, 2); ?></div>
                    <div style="font-size: 0.9rem; color: #94a3b8;">Total Payouts</div>
                </div>
            </div>
            
            <!-- Top Referrers Table -->
            <table class="users-table">
                <tr>
                    <th>Username</th>
                    <th>Referral Code</th>
                    <th>Total Referrals</th>
                    <th>Total Earnings</th>
                </tr>
                <?php if (empty($referral_stats)): ?>
                    <tr>
                        <td colspan="4" style="text-align: center; color: #94a3b8;">No referral data available</td>
                    </tr>
                <?php else: ?>
                    <?php foreach ($referral_stats as $stat): ?>
                    <tr>
                        <td><?php echo htmlspecialchars($stat['username']); ?></td>
                        <td><code style="background: rgba(34, 197, 94, 0.1); padding: 4px 8px; border-radius: 4px; color: #22c55e;"><?php echo htmlspecialchars($stat['referral_code']); ?></code></td>
                        <td><?php echo htmlspecialchars($stat['total_referrals']); ?></td>
                        <td>₹<?php echo number_format($stat['total_referral_earnings'], 2); ?></td>
                    </tr>
                    <?php endforeach; ?>
                <?php endif; ?>
            </table>
        </div>
        
        <div class="admin-box admin-section" id="add-admin" style="min-width:500px;">
            <div class="settings-section-title">Add Admin</div>
            <div class="settings-placeholder">Yahan se aap <a href="admin_register.php">naya admin</a> add kar sakte hain.</div>
        </div>
    </div>
    <script>
    // Sidebar SPA logic
    document.querySelectorAll('.admin-sidebar-link[data-section]').forEach(function(link) {
        link.addEventListener('click', function(e) {
            e.preventDefault();
            // Remove active from all
            document.querySelectorAll('.admin-sidebar-link').forEach(l => l.classList.remove('active'));
            link.classList.add('active');
            // Hide all sections
            document.querySelectorAll('.admin-section').forEach(s => s.classList.remove('active'));
            // Show selected
            var sec = document.getElementById(link.getAttribute('data-section'));
            if (sec) sec.classList.add('active');
            // Update hash
            window.location.hash = link.getAttribute('href');
        });
    });
    // On page load, show section from hash
    window.addEventListener('DOMContentLoaded', function() {
        var hash = window.location.hash.replace('#','');
        if (hash && document.getElementById(hash)) {
            document.querySelectorAll('.admin-sidebar-link').forEach(l => l.classList.remove('active'));
            document.querySelectorAll('.admin-section').forEach(s => s.classList.remove('active'));
            var link = document.querySelector('.admin-sidebar-link[data-section="'+hash+'"]');
            var sec = document.getElementById(hash);
            if (link) link.classList.add('active');
            if (sec) sec.classList.add('active');
        }
    });
    </script>
</body>
</html> 